
// Copyright (c) 2003 Sonic Foundry, Inc. and Sonic Foundry 
// Media Systems, Inc. Neither this code nor any portion 
// thereof maybe reproduced, altered,or otherwise changed, 
// distributed or copied,without the express written 
// permission of Sonic Foundry. 
// All rights reserved.

// BEGINFILE WM64PlayerArea.js ----------------------------------------------------------------------->

WM64PlayerArea.prototype = new PlayerArea();
function WM64PlayerArea(container, containingWindow, ID)
{
	this.m_debugLevel = SfDebug.Verbose;
//	this.m_debugLevel = SfDebug.Information;

	this.PlayerType = PlayerType.WM64;
	this.EnableVolumeControls = null; // from cs code
	this.PlayerEventManager = new PlayerEventManager(this);

	// Need to add the following line because
	// when you call .prototype this gets executed	
	if (container && containingWindow && ID)
	{
		this.InitializeArea(container, containingWindow, ID);
	}
	
	this.LoadPlayerControl = function()
	{
		var objEmbeddedPlayer = this.GetEmbeddedPlayer();
		this.pci = new PlayerControlWM64(objEmbeddedPlayer);
	}
	
	this.SetupPlayer = function()
	{
		this.LoadPlayerControl();
		
		// initialize PlayerEventManager
		this.PlayerEventManager.AttachEmbeddedPlayerEvents(this.pci.objEmbedded);

 		// create the position slider
		this.PositionTimer = null;

		this.AddEventHandlers();

		this.pci.SetMedia(MainHelper.Presentation.VideoUrl);
		
		if (this.PlayFrom != -1)
		{
			this.SetPlayFrom(this.PlayFrom/1000);
		}

		this.Debug("Loaded: " + MainHelper.Presentation.VideoUrl + " Duration: " + this.pci.GetMediaDuration());
		this.PostMediaLengthObtainedEvent();
	    
		this.TimedUpdate();
    
		MainHelper.EventPlayerSetupComplete.Post(new SfEventArgs());
		this.Volume = new Volume(this.pci);
		this.Volume.InitializeVolume();
 		this.EnableTransportButtons(PlayState.Playing);
	}
	
	// this is overridden for WM7
	this.SetPlayFrom = function(timeInSecs)
	{
		this.Debug("SetPlayFrom: " + timeInSecs);

		setTimeout(this.Container + ".pci.SetPosition(" + timeInSecs + ");", 1000);
	}
	
	this.AddEventHandlers = function()
	{
		this.SliderNotifyEventHandler = new SfEventHandler("Player");
		this.SliderNotifyEventHandler.Container = this.Container;
		this.SliderNotifyEventHandler.MethodName = "OnSliderNotifyEvent";
		MainHelper.EventSliderNotify.AddHandler(this.SliderNotifyEventHandler);
    
 		this.ScriptEventHandler = new SfEventHandler("Player");
		this.ScriptEventHandler.MethodName = "OnScriptEvent";
		this.ScriptEventHandler.Container = this.Container;
		MainHelper.EventScript.AddHandler(this.ScriptEventHandler);

 		this.CommandEventHandler = new SfEventHandler("Player");
		this.CommandEventHandler.MethodName = "OnCommandEvent";
		this.CommandEventHandler.Container = this.Container;
		MainHelper.EventCommand.AddHandler(this.CommandEventHandler);
		
		this.EventManagerPlayStateChangeHandler = new SfEventHandler();
		this.EventManagerPlayStateChangeHandler.Container = this.Container;
		this.EventManagerPlayStateChangeHandler.MethodName = "OnEventManagerPlayStateChangeEvent";
		MainHelper.EventPlayerPlayStateChanged.AddHandler(this.EventManagerPlayStateChangeHandler);
    
		this.EventManagerPositionChangeHandler = new SfEventHandler();
		this.EventManagerPositionChangeHandler.MethodName = "OnEventManagerPositionChangeEvent";
		this.EventManagerPositionChangeHandler.Container = this.Container;
		MainHelper.EventPlayerPositionChanged.AddHandler(this.EventManagerPositionChangeHandler);
	}
	
	this.RemoveEventHandlers = function()
	{
		MainHelper.EventSliderNotify.RemoveHandler(this.SliderNotifyEventHandler);
		MainHelper.EventScript.RemoveHandler(this.ScriptEventHandler);
		MainHelper.EventCommand.RemoveHandler(this.CommandEventHandler);
		MainHelper.EventPlayerPlayStateChanged.RemoveHandler(this.EventManagerPlayStateChangeHandler);
		MainHelper.EventPlayerPositionChanged.RemoveHandler(this.EventManagerPositionChangeHandler);
	}
	
	this.OnUnLoad = function()
	{
		this.RemoveEventHandlers();
	}

	this.PostMediaLengthObtainedEvent = function()
	{
		var range = new Object();
		range.Left = 0;
		range.Right = this.pci.GetMediaDuration();
		
		MainHelper.EventPlayerMediaLengthObtained.Post(range);
	}
  
    this.EnableTransportButtons = function(state)
    {
		this.Debug("EnableTransportButtons called, state: " + GetPlayStateName(state));
		MainHelper.EventPlayerStateChanged.Post(state);
    }
    
    
	this.Play = function()
	{
		this.Debug("Play called");
		if (this.pci.GetPlayState() == PlayState.Stopped)
		{
			// reset presentation from stop
			var currentPos = this.pci.GetPosition();
			this.Debug('currentPosition: ' + currentPos);
			if (currentPos == 0)
			{
				MainHelper.CurrentSlideNumber = -1;
				MainHelper.EventPlayBegin.Post(new Object());
			}
		}
		this.pci.Play();
		this.EnableTransportButtons(PlayState.Playing);

		MainHelper.PresentationEnded = false;

		this.PositionTimer = setTimeout(this.Container + ".TimedUpdate()", 500);
	}

	this.Pause = function()
	{
		if (this.pci.GetPlayState() == PlayState.Playing)
		{
			this.pci.Pause();
			this.EnableTransportButtons(PlayState.Paused);

			// no need for timedupdate now
			if (this.PositionTimer)
			{
				clearTimeout(this.PositionTimer);
			}
		}
	}

	this.Stop = function()
	{

		if (this.pci.GetPlayState() == PlayState.Playing)
		{
			this.pci.Stop();
			this.EnableTransportButtons(PlayState.Stopped);

			// no need for timedupdate now
			if (this.PositionTimer)
			{
				clearTimeout(this.PositionTimer);
			}
		}

	}

	this.FullScreen = function()
	{
		if (this.pci.GetPlayState() != PlayState.Stopped)
		{
			this.pci.SetFullScreen(true);
		}
	}

	this.ToggleMute = function()
	{
		this.Debug("ToggleMute called");
		if (this.Volume == null)
		{
			this.Debug("Volume not initialized yet");
			return;
		}
		this.Volume.ToggleMute();
	}
	
	this.SetMute = function(mute)
	{
		this.Debug("SetMute()");
		if (this.Volume == null)
		{
			this.Debug("Volume not initialized yet");
			return;
		}
		this.Volume.SetMute(mute);
	}
	
	this.VolumeUp = function()
	{
		this.Debug("VolumeUp called");
		if (this.Volume == null)
		{
			this.Debug("Volume not initialized yet");
			return;
		}
		this.Volume.ChangeVolume(6);
	}

	this.VolumeDown = function()
	{
		this.Debug("VolumeDown called");
		if (this.Volume == null)
		{
			this.Debug("Volume not initialized yet");
			return;
		}
		this.Volume.ChangeVolume(-6);
	}

	this.OnScriptEvent = function(args)
	{
		switch(args.Command)
		{
			case SfScriptCommandType.EndPresentation:
				this.Debug("OnEndPresentation");
				this.pci.Stop();
				this.EnableTransportButtons(PlayState.Stopped);
				MainHelper.PresentationEnded = true;
				break;
		}
	}

	function WM64PlayerArea.prototype.OnCommandEvent(args)
	{
		switch(args.Command)
		{
			case SfCommandType.NavigateToSlide:
				var slideNumber = args.SlideNumber;
				if (!slideNumber)
				{
					SfDebug.DPF(SfDebug.ErrMsgCritical, "Navigate to Slide no SlideNumber present");
					return;
				}
				this.NavigateToSlide(Number(slideNumber));
				break;
			case SfCommandType.Play:
				this.Play();
				break;
			case SfCommandType.Pause:
				this.Pause();
				break;
			case SfCommandType.Stop:
				this.Stop();
				break;
			case SfCommandType.FullScreen:
				this.FullScreen();
				break;
			case SfCommandType.ToggleMute:
				this.ToggleMute();
				break;
			case SfCommandType.Mute:
				this.SetMute(true);
				break;
			case SfCommandType.UnMute:
				this.SetMute(false);
				break;
			case SfCommandType.VolumeUp:
				this.VolumeUp();
				break;
			case SfCommandType.VolumeDown:
				this.VolumeDown();
				break;
		}
	}

	this.OnSliderNotifyEvent = function(objNotify)
	{
		switch(objNotify.NotifyType)
		{
			case SfSliderNotifyType.NewPosition:
				this.Debug("OnSliderNotify called: " + objNotify.Position);
				this.pci.SetPosition(objNotify.Position);
				break;
		}
	}
	
	this.OnEventManagerPlayStateChangeEvent = function(args)
	{
		this.Debug("OnEventManagerPlayStateChangeEvent, playState: " + GetPlayStateName(args.PlayState));
		this.EnableTransportButtons(args.PlayState);
		
		switch(args.PlayState)
		{
			case PlayState.Buffering:
				this.PostMediaLengthObtainedEvent();
				break;
					
			case PlayState.Playing:
				// local files won't have duration until now
				this.Debug("Resetting duration " + this.pci.GetMediaDuration());
				this.PostMediaLengthObtainedEvent();
				break;
		}
	}

	this.SuppressNewSlideEvent = false;
	this.OnEventManagerPositionChangeEvent = function(args)
	{
		this.Debug("OnEventManagerPositionChangeEvent called, newPosition: " + args.NewPosition);
		if (this.SuppressNewSlideEvent == true)
		{
			this.SuppressNewSlideEvent = false;
			return;
		}

		// fire event to change the slide being shown now.
		var slideNumber = this.CalculateSlideNumberToShow(args.NewPosition);
		
		if (slideNumber == 0)
		{
			this.Debug("no need to change slide");
			return;
		}
		
		this.Debug("switching to slide: " + slideNumber);
		var args =  MainHelper.CreateShowSlideEventArgs(slideNumber);
		MainHelper.EventScript.Post(args);	
	}

	this.TimedUpdate = function()
	{
		// !!! don't post when user is sliding
		this.PostPlayerTimerUpdatedEvent(this.pci.GetPosition());

		this.PositionTimer = setTimeout(this.Container + ".TimedUpdate()", 500);
	}
	
	this.PostPlayerTimerUpdatedEvent = function(pos)
	{
		MainHelper.EventPlayerTimerUpdated.Post(pos);
	}
	
	this.NavigateToSlide = function(slideNumber)
	{
		this.Debug("NavigateToSlide: " + slideNumber + " called");
		if (MainHelper.Presentation.Status != PresentationStatus.ReplayReady)
		{
			this.Debug("can not navigate when presentation is not on demand");
			return;
		}
		
		if (slideNumber < 0)
		{
			this.Debug("SlideNumber is less than 0");
			return;
		}
		if (MainHelper.MaxSlideTimings < slideNumber)
		{
			SfDebug.DPF(SfDebug.ErrMsgCritical, "slidetimings not known for slideNumber: " + slideNumber);
			return;
		}
		
		var slideTiming = MainHelper.Presentation.SlideTimings[slideNumber-1];
	
		var timeCode = (slideNumber == 0) ? 0.00 : slideTiming.Time;
		this.Debug("Navigating to timeCode: " + timeCode);
		
		this.pci.SetPosition(timeCode);
		this.SuppressNewSlideEvent = true;
		this.PostPlayerTimerUpdatedEvent(timeCode);

		// event is not automatically getting fired
		// so do it manually
		var args = MainHelper.CreateShowSlideEventArgs(slideNumber);
		this.ScriptParser.NotifyScriptEvent(args);

		MainHelper.EventCommand.Post(new CommandArgs(SfCommandType.Play));

		MainHelper.EventCommand.Post(new CommandArgs(SfCommandType.ShowSlideShow));
	
	}
}

// ENDFILE WM64PlayerArea.js ------------------------------------------------------------------------->

// BEGINFILE PlayerControlWM64.js -------------------------------------------------------------------->

    var WM6OpenState =
    {
        Closed:     0,
        LoadingASX: 1,
        LoadingNSC: 2,
        Locating:   3,
        Connecting: 4,
        Opening:    5,
        Open:       6
    }
    
    var WM6ReadyState=
    {
		Uninitialized:	0,
		Loading:		1,
		Interactive:	3,
		Complete:		4
	}
    
    var WM6PlayState =
    {
        Stopped:        0,
        Paused:         1,
        Playing:        2,
        Waiting:        3,
        ScanForward:    4,
        ScanReverse:    5,
        SkipForward:    6,
        SkipReverse:    7,
        Closed:         8
    }

    var WM6DisplaySize =
    {
        DefaultSize:        0,
        HalfSize:           1,
        DoubleSize:         2,
        FullScreen:         3,
        FitToSize:          4,
        OneSixteenthScreen: 5,
        OneFourthScreen:    6,
        OneHalfScreen:      7
    }
    

function PlayerControlWM64(objEmbeddedPlayer)
{
	this.objEmbedded=objEmbeddedPlayer;
	
	this.GetVersionInfo = function()
	{
		return "6.4.0.0";
	}

	this.Play = function()
	{
		this.objEmbedded.Play();
	}
	
	this.Stop = function()
	{
		this.objEmbedded.Stop();
	}
	
	this.Pause = function()
	{
		this.objEmbedded.Pause();
	}
	
	
	this.ConvertOpenState = function(WM6State)
	{
		// need to translate this state
		
		switch(WM6State)
		{
		        
            case WM6OpenState.LoadingASX:
                return OpenState.PlaylistLoading;
            
            case WM6OpenState.Locating:
                return OpenState.MediaLocating;
                
            case WM6OpenState.Connecting:
                return OpenState.MediaConnecting;
                
            case WM6OpenState.Opening:
                return OpenState.MediaOpening;
                
            case WM6OpenState.Open:
                return OpenState.MediaOpen;
                
            case WM6OpenState.Closed:
            case WM6OpenState.LoadingNSC:
            default:
                return OpenState.Undefined;
		}

	}
	
	this.ConvertPlayState = function(WM6State)
	{
		// need to translate this state
		
		switch(WM6State)
		{
		
		    case WM6PlayState.Stopped:
		        return PlayState.Stopped;
		        
            case WM6PlayState.Paused:
                return PlayState.Paused;
                
            case WM6PlayState.Playing:
                return PlayState.Playing;
            
            case WM6PlayState.ScanForward:
                return PlayState.ScanForward;
                
            case WM6PlayState.ScanReverse:
                return PlayState.ScanReverse;
                
            case WM6PlayState.Waiting:
                return PlayState.Waiting;
                
            case WM6PlayState.SkipForward:
            case WM6PlayState.SkipReverse:
            case WM6PlayState.Closed:
            default:
                return PlayState.Undefined;
            // wm6 doesn't have Buffering,MediaEnded,Transitioning,Ready
		}

	}
	
	
	this.GetPlayState = function()
	{
		// need to translate this state
		
		var WM6State = this.objEmbedded.playState;
		
		return this.ConvertPlayState(WM6State);

	}
	
	this.SetFullScreen = function(f)
	{
		if (f)
		{
			this.objEmbedded.DisplaySize = WM6DisplaySize.FullScreen;
		}
		else
		{
			this.objEmbedded.DisplaySize = WM6DisplaySize.DefaultSize;
		}

	}
	
	this.SetPosition = function(dPosition)
	{
		this.objEmbedded.CurrentPosition=dPosition;
	}

	this.GetPosition = function()
	{
		return this.objEmbedded.CurrentPosition;
	}
	
	this.GetMediaDuration = function()
	{
		return this.objEmbedded.Duration;
	}

	this.SetMedia = function(sMedia)
	{
		this.objEmbedded.Filename=sMedia;
	}
	
	
	this.GetCurrentMarker = function()
	{
		return this.objEmbedded.CurrentMarker;
	}
	
	this.GetMediaMarkerCount = function()
	{
		return this.objEmbedded.MarkerCount;
	}
	
	this.GetMediaMarkerName = function(nIndex)
	{

		return this.objEmbedded.GetMarkerName(nIndex);
	}
	
	this.GetMediaMarkerTime = function(nIndex)
	{
		return this.objEmbedded.GetMarkerTime(nIndex);
	}
	
	this.GetVolume = function()
	{
		// need to translate
		// convert -1700-0 to 0-100
		//
		
		var nVolume = this.objEmbedded.volume;
		
		
		nVolume/=17;
		nVolume+=100;
		
		if (nVolume<0)
		    nVolume=0;
		    
		if (nVolume>100)
		    nVolume=100;

		return nVolume;
	}
	
	this.SetVolume = function(nVolume)
	{
		// need to translate
		// convert 0-100 to -1700-0
		//
		
		    
		if (nVolume<0)
		    nVolume=0;
		if (nVolume>100)
		    nVolume=100;
		    
		nVolume-=100;
		nVolume*=17;
		
		this.objEmbedded.volume=nVolume;
	}
	
	this.SetMute = function(fMute)
	{
		this.objEmbedded.Mute=fMute;
	}
	
	this.GetMute = function()
	{
		return this.objEmbedded.Mute;
	}
		
}

// ENDFILE PlayerControlWM64.js ---------------------------------------------------------------------->

// BEGINFILE PlayerEventManager.js ------------------------------------------------------------------>

function PlayerEventManager(playerArea)
{
	this.m_debugLevel = SfDebug.Verbose;
//	this.m_debugLevel = SfDebug.Information;
	
	this.lastPlayState = -1;

	// attachevent loses the object information
	// just the method information 
	// therefore "this" is not used in the class
	this.AttachEmbeddedPlayerEvents = function(objPlayer)
	{
		AttachEvent(objPlayer, "ScriptCommand", this.OnScriptCommand);

		AttachEvent(objPlayer, "PlayStateChange", this.OnPlayStateChange);
		AttachEvent(objPlayer, "PositionChange", this.OnPositionChange);
		AttachEvent(objPlayer, "OpenStateChange", this.OnOpenStateChange);
		
		var ver = parseInt(PlayerAreaInstance.pci.GetVersionInfo());

		if (ver<7)
		{	
			AttachEvent(objPlayer, "Buffering", this.OnBuffering);
			AttachEvent(objPlayer, "NewStream", this.OnNewStream);
			AttachEvent(objPlayer, "EndOfStream", this.OnEndOfStream);
		}	
	}

	this.OnScriptCommand = function(sType, sParam)
	{
		PlayerAreaInstance.ScriptParser.ParseScriptFromStream(sType, sParam);
	}

	this.OnPlayStateChange = function(newState)
	{
		
		// normal call passes newstate
		// 6.4 passes oldstate newstate null
		// 7  passes newstate null
		PlayerAreaInstance.PlayerEventManager.Debug("OnPlayStatechanged, newState: " + newState);

		if (arguments.length>2)
		{
			newState = arguments[1];

			if (PlayerAreaInstance.pci.ConvertPlayState)
			{
				newState = PlayerAreaInstance.pci.ConvertPlayState(newState);
			}
		}
		
		PlayerAreaInstance.PlayerEventManager.Debug("PlayStateChange New: " + GetPlayStateName(newState) + "("+newState+")");
		PlayerAreaInstance.PlayerEventManager.UpdateLastPlayState(newState);
		PlayerAreaInstance.PlayerEventManager.NotifyPlayStateChange(newState);
	}


	this.OnPositionChange = function(dblOldPosition, dblNewPosition)
	{
		PlayerAreaInstance.PlayerEventManager.Debug("PositionChange Old: " + dblOldPosition + " New: " + dblNewPosition);
		PlayerAreaInstance.PlayerEventManager.NotifyPositionChange(dblOldPosition, dblNewPosition);
	}

	this.OnOpenStateChange = function(openState)
	{
		
		if (PlayerAreaInstance.pci.ConvertOpenState)
		{
			openState = PlayerAreaInstance.pci.ConvertOpenState(openState);
		}

		PlayerAreaInstance.PlayerEventManager.Debug("OpenState " + openState);

	}

	this.OnBuffering = function(bStart)
	{
		PlayerAreaInstance.PlayerEventManager.Debug("Buffering: " + bStart);
	    
		if (bStart)
		{
			if (PlayerAreaInstance.PlayerEventManager.UpdateLastPlayState(PlayState.Buffering))
			{
				PlayerAreaInstance.PlayerEventManager.OnPlayStateChange(PlayState.Buffering);
			}
		}
		else
		{
			if (PlayerAreaInstance.PlayerEventManager.UpdateLastPlayState(PlayState.Playing))
			{
				PlayerAreaInstance.PlayerEventManager.OnPlayStateChange(PlayState.Playing);
			}
		}
	}

	this.OnEndOfStream = function(lResult)
	{
		PlayerAreaInstance.PlayerEventManager.Debug("EndofStream: " + lResult);
	    
		if (PlayerAreaInstance.PlayerEventManager.UpdateLastPlayState(PlayState.MediaEnded))
		{
			PlayerAreaInstance.PlayerEventManager.OnPlayStateChange(PlayState.MediaEnded);
		}
	}

	this.UpdateLastPlayState = function(newState)
	{
		PlayerAreaInstance.PlayerEventManager.Debug("UpdateLastPlayState: last: " + this.lastPlayState + ", new: " + newState);
		if (newState != this.lastPlayState)
		{
			this.lastPlayState = newState;
			return true;
		}
		return false;
	}

	this.OnNewStream = function()
	{
		PlayerAreaInstance.PlayerEventManager.Debug("NewStream ");
		if (PlayerAreaInstance.PlayerEventManager.UpdateLastPlayState(PlayState.Transitioning))
		{
			PlayerAreaInstance.PlayerEventManager.OnPlayStateChange(PlayState.Transitioning);
		}
	}

	// Notification functions	
	this.NotifyPlayStateChange = function(playState)
	{
		argPost = new SfEventArgs();
		
		argPost.PlayState = playState;
		MainHelper.EventPlayerPlayStateChanged.Post(argPost);
	}
	
	this.NotifyPositionChange = function(oldPosition, newPosition)
	{
		argPost = new SfEventArgs();
		
		argPost.OldPosition = oldPosition;
		argPost.NewPosition = newPosition;
		MainHelper.EventPlayerPositionChanged.Post(argPost);
	
	}
	
	this.Debug = function(str)
	{
		SfDebug.DPF(this.m_debugLevel, "PlayerEventManager: " + str);
	}
}

// ENDFILE PlayerEventManager.js -------------------------------------------------------------------->


// BEGINFILE PlayerPositionArea.js ------------------------------------------------------------------>


PlayerPositionArea.prototype = new AreaBase();
function PlayerPositionArea(container, containingWindow, ID)
{
	this.m_debugLevel = SfDebug.Verbose;
//	this.m_debugLevel = SfDebug.Information;

	this.InitializeArea(container, containingWindow, ID);

	this.Debug = function(msg)
	{
		SfDebug.DPF(this.m_debugLevel, "PlayerPositionArea: " + msg);
	}
	
	this.OnLoad = function()
	{
		this.Debug("OnLoad");
		this.Initialize();
		this.AddEventHandlers();
	}
	
	this.AddEventHandlers = function()
	{
		this.Debug("AddEventHandlers");
		this.MediaLengthObtainedEventHandler = new SfEventHandler("SliderArea");
		this.MediaLengthObtainedEventHandler.MethodName = "OnMediaLengthObtained";
		this.MediaLengthObtainedEventHandler.Container = this.Container;
		MainHelper.EventPlayerMediaLengthObtained.AddHandler(this.MediaLengthObtainedEventHandler);

		this.SliderNotifyEventHandler = new SfEventHandler("PlayerPositionArea");
		this.SliderNotifyEventHandler.MethodName = "OnSliderNotify";
		this.SliderNotifyEventHandler.Container = this.Container;
		MainHelper.EventSliderNotify.AddHandler(this.SliderNotifyEventHandler);

		this.PlayerTimerUpdatedEventHandler = new SfEventHandler("PlayerPositionArea");
		this.PlayerTimerUpdatedEventHandler.MethodName = "OnPlayerTimerUpdated";
		this.PlayerTimerUpdatedEventHandler.Container = this.Container;
		MainHelper.EventPlayerTimerUpdated.AddHandler(this.PlayerTimerUpdatedEventHandler);
	}
	
	this.RemoveEventHandlers = function()
	{
		this.Debug("RemoveEventHandlers");
		MainHelper.EventPlayerMediaLengthObtained.RemoveHandler(this.MediaLengthObtainedEventHandler);
		MainHelper.EventSliderNotify.RemoveHandler(this.MediaLengthObtainedEventHandler);
		MainHelper.EventPlayerTimerUpdated.RemoveHandler(this.PlayerTimerUpdatedEventHandler);
	}
	
	this.OnMediaLengthObtained = function(range)
	{
		this.Left = range.Left;
		this.Right = range.Right;
	}
	
	this.OnSliderNotify = function(notifyArgs)
	{
		switch(notifyArgs.NotifyType)
		{
			case SfSliderNotifyType.DragPosition:
				this.UpdatePositionStatus(notifyArgs.Position);
				break;
		}
	}
	
	this.OnPlayerTimerUpdated = function(position)
	{
		if (MainHelper.UserIsSliding)
		{
			return;
		}
		this.UpdatePositionStatus(position); 
	}
	
	this.UpdatePositionStatus = function(position)
	{
		this.MyDiv.innerHTML = this.FormatPosition(position) + "/" + this.FormatPosition(this.Right);
	}
	
	this.FormatPosition = function(pos)
	{
		var tmp = pos;
		var mm = Math.floor(tmp/60);
		var ss = Math.floor(pos-(mm*60));
		
		var sTime = "";
		
		if (mm < 10)
		{
			sTime += "0"+mm;
		}
		else
		{
			sTime += mm;
		}
			
		sTime += ":";
		
		if (ss < 10)
		{
			sTime += "0" + ss;
		}
		else
		{
			sTime += ss;
 		}
			
		return sTime;
	}

	this.OnUnLoad = function()
	{
		this.Debug("OnUnLoad");
		this.RemoveEventHandlers();
	}
	
	this.Initialize = function()
	{
		this.Debug("Initialize called");
		this.Left = 0;
		this.Right = 100;
		this.MyDiv = this.GetDiv();
	}
}

// ENDFILE PlayerPositionArea.js -------------------------------------------------------------------->

// BEGINFILE PlayerStatusArea.js -------------------------------------------------------------------->


PlayerStatusArea.prototype = new AreaBase();
function PlayerStatusArea(container, containingWindow, ID)
{
	this.m_debugLevel = SfDebug.Verbose;
//	this.m_debugLevel = SfDebug.Information;

	this.InitializeArea(container, containingWindow, ID);

	this.Debug = function(msg)
	{
		SfDebug.DPF(this.m_debugLevel, "PlayerStatusArea: " + msg);
	}
	
	this.OnLoad = function()
	{
		this.Debug("OnLoad");
		this.Initialize();
		this.AddEventHandlers();
	}
	
	this.AddEventHandlers = function()
	{
		this.Debug("AddEventHandlers");
		this.PlayerStateChangedEventHandler = new SfEventHandler("PlayButtonArea");
		this.PlayerStateChangedEventHandler.MethodName = "OnPlayerStateChangedEvent";
		this.PlayerStateChangedEventHandler.Container = this.Container;
		MainHelper.EventPlayerStateChanged.AddHandler(this.PlayerStateChangedEventHandler);
	}
	
	this.RemoveEventHandlers = function()
	{
		this.Debug("RemoveEventHandlers");
		MainHelper.EventPlayerStateChanged.RemoveHandler(this.PlayerStateChangedEventHandler);
	}
	
	this.OnPlayerStateChangedEvent = function(state)
	{
		this.UpdateStatus(state);
	}
	
	this.UpdateStatus = function(status)
	{
		var name = GetPlayStateName(status);
		this.Debug("Changing status to: " + name);
		this.MyDiv.innerHTML = name;
	}
	
	this.OnUnLoad = function()
	{
		this.Debug("OnUnLoad");
		this.RemoveEventHandlers();
	}
	
	this.Initialize = function()
	{
		this.Debug("Initialize called");
		this.Left = 0;
		this.Right = 100;
		this.MyDiv = this.GetDiv();
	}
}

// ENDFILE PlayerStatusArea.js ---------------------------------------------------------------------->

// BEGINFILE Volume.js ------------------------------------------------------------------------------>


function Volume(pci)
{
	this.IsMuted = false;
	this.PreviousVolume = null;
	this.pci = pci;
	this.m_debugLevel = SfDebug.Verbose;
	

	this.InitializeVolume = function()
	{
		var currentVolume = this.GetVolumeFromPlayer();
		var curIdx = Volume.ToIndex(currentVolume);
		
		this.PostVolumeUpDownEvent(curIdx);
	}
		
	this.GetVolumeFromPlayer = function()
	{
	    if (this.pci == null)
	    {
	        return 0;
	    }
	        
	    return this.pci.GetVolume();
	}
	
	
	this.ToggleMute = function()
	{
		this.SetMute(!this.IsMuted);
	}
	
	this.SetMute = function(mute)
	{
	    if (mute == false)
	    {
	    	this.pci.SetMute(false);
	    	
	    	if (this.PreviousVolume != null)
			{
				this.pci.SetVolume(this.PreviousVolume);
	        }
	        
	        this.IsMuted = false;
	        this.PostMuteEvent(SfVolumeChangeType.UnMuted);
	    }
	    else
	    {
			this.PreviousVolume = this.pci.GetVolume();
	    	this.pci.SetMute(true);
	        
	        this.IsMuted=true;
	        this.PostMuteEvent(SfVolumeChangeType.Muted);
	    }
	}
	
	this.ChangeVolume = function(deltaLevel)
	{
		var currentVolume = this.GetVolumeFromPlayer();
		var curIdx = Volume.ToIndex(currentVolume);
		var newIdx = Volume.ToIndex(currentVolume + deltaLevel);
	
		var fMute=false;
		
		if (this.IsMuted && (deltaLevel>0 ||(curIdx!=0)))
		{
			this.ToggleMute();
		}
		
		this.Debug("Curridx:"+curIdx+" NewIDX:"+newIdx);
		
		if (deltaLevel<0)
		{
			// if we're already minned
			if (curIdx==0)
			{
				return;
			}
			
			if (newIdx==0)
			{
				fMute=true;
			}
		}
		else
		{
			// if we're already maxed out
			if (curIdx>=(Volume.Positions-1))
			{
				return;
			}
		}
		
		currentVolume += deltaLevel;
		
		this.Debug("Setting Volume: "+currentVolume);
		this.pci.SetVolume(currentVolume);
		
		// must mute after setting volume.. WMP won't let you set the volume
		// while it's muted
		if (fMute)
		{
			if (!this.IsMuted)
			{
				this.ToggleMute();
			}
		}
		
		this.PostVolumeUpDownEvent(newIdx);
	}
	
	this.PostVolumeUpDownEvent = function(index)
	{
		var args = new VolumeChangedArgs(SfVolumeChangeType.VolumeUpDown);
		args.VolumeIndex = index;
		MainHelper.EventVolumeChanged.Post(args);
	}
		
	this.PostMuteEvent = function(muteType)
	{
		MainHelper.EventVolumeChanged.Post(new VolumeChangedArgs(muteType));
	}
	
	this.Debug = function(msg)
	{
		SfDebug.DPF(this.m_debugLevel, "Volume: " + msg);
	}	
}


function VolConverter(dbLevel, index)
{
	this.Volume = dbLevel;
	this.BarIndex = index;
}


Volume.Positions=18;

Volume.ConverterArray= new Array(Volume.Positions);
Volume.ConverterArray[17]=(new VolConverter(100,17));
Volume.ConverterArray[16]=(new VolConverter(96,16));
Volume.ConverterArray[15]=(new VolConverter(90,15));
Volume.ConverterArray[14]=(new VolConverter(84,14));
Volume.ConverterArray[13]=(new VolConverter(78,13));
Volume.ConverterArray[12]=(new VolConverter(72,12));
Volume.ConverterArray[11]=(new VolConverter(66,11));
Volume.ConverterArray[10]=(new VolConverter(60,10));
Volume.ConverterArray[9]=(new VolConverter(54,9));
Volume.ConverterArray[8]=(new VolConverter(48,8));
Volume.ConverterArray[7]=(new VolConverter(42,7));
Volume.ConverterArray[6]=(new VolConverter(36,6));
Volume.ConverterArray[5]=(new VolConverter(30,5));
Volume.ConverterArray[4]=(new VolConverter(24,4));
Volume.ConverterArray[3]=(new VolConverter(18,3));
Volume.ConverterArray[2]=(new VolConverter(12,2));
Volume.ConverterArray[1]=(new VolConverter(6,1));
Volume.ConverterArray[0]=(new VolConverter(0,0));


Volume.ToIndex=function(lVolume)
{
	var i;
	
	for (i=0; i<Volume.ConverterArray.length; i++)
	{
		if (lVolume<=Volume.ConverterArray[i].Volume)
		{
			return Volume.ConverterArray[i].BarIndex;
		}
	}
	
	return Volume.ConverterArray.length-1;

}

// ENDFILE Volume.js -------------------------------------------------------------------------------->

// BEGINFILE VolumeBarArea.js ----------------------------------------------------------------------->

VolumeBarArea.prototype = new AreaBase();
function VolumeBarArea(container, containingWindow, ID)
{
	this.m_debugLevel = SfDebug.Verbose;
//	this.m_debugLevel = SfDebug.Information;
	this.ImageBase = null;
	this.ImageElement = null;
	this.VolumeIndex = 0;

	this.InitializeArea(container, containingWindow, ID);

	this.Debug = function(msg)
	{
		SfDebug.DPF(this.m_debugLevel, "VolumeBar: " + msg);
	}
	
	this.OnLoad = function()
	{
		this.Debug("OnLoad");
		this.Initialize();
		this.AddEventHandlers();
	}
	
	this.AddEventHandlers = function()
	{
		this.Debug("AddEventHandlers");
 		this.VolumeChangedEventHandler = new SfEventHandler("VolumeBarArea");
		this.VolumeChangedEventHandler.MethodName = "OnVolumeChangedEvent";
		this.VolumeChangedEventHandler.Container = this.Container;
		MainHelper.EventVolumeChanged.AddHandler(this.VolumeChangedEventHandler);
	}
	
	this.OnVolumeChangedEvent = function(args)
	{
		this.Debug("OnVolumeChangedEvent: " + args);
		switch(args.ChangeType)
		{
			case SfVolumeChangeType.VolumeUpDown:
				this.ChangeVolumeIndex(args.VolumeIndex);
				break;
		}
	}
	
	this.ChangeVolumeIndex = function(index)
	{
		this.Debug("Change volume index called");
		if (!index)
		{
			this.Debug("invalid index");
		}
		this.VolumeIndex = index;
		this.ImageElement.src = this.ImageBase + "/bar" + index + ".gif";
	}
	
	this.RemoveEventHandlers = function()
	{
		this.Debug("RemoveEventHandlers");
		MainHelper.EventVolumeChanged.RemoveHandler(this.VolumeChangedEventHandler);
	}
	
	this.OnUnLoad = function()
	{
		this.Debug("OnUnLoad");
		this.RemoveEventHandlers();
	}
	
	this.Initialize = function()
	{
		this.Debug("Initialize called");
		this.ImageElement = SfDOM.FindElementFromID(document, this.ID + "Image");
		if (this.ImageElement == null)
		{
			this.Debug("Image element not found!!");
		}
	}

}

// ENDFILE VolumeBarArea.js ------------------------------------------------------------------------->

// BEGINFILE SamiDropDownArea.js -------------------------------------------------------------------->
SamiDropDownArea.prototype = new AreaBase();
function SamiDropDownArea(container, containingWindow, ID)
{
	this.m_debugLevel = SfDebug.Verbose;
//	this.m_debugLevel = SfDebug.Information;

	this.InitializeArea(container, containingWindow, ID);

	this.IsClosedCaptioningVisible = false;
	
	this.OnLoad = function()
	{
		this.Debug("OnLoad");
		this.ToggleA = SfDOM.FindElementFromID(document, this.ID + "Toggle");
		this.CaptioningContainer = SfDOM.FindElementFromID(document, this.ID + "Container");
		this.ToggleA.onclick = new Function("", this.Container + ".ToggleClosedCaptioning();");
	}
	
	this.ToggleClosedCaptioning = function() 
	{
		if ( this.IsClosedCaptioningVisible ) 
		{
			this.CaptioningContainer.style.display = 'none';
			this.ToggleA.innerText = '^';
		} 
		else 
		{
			this.CaptioningContainer.style.display = '';
			this.ToggleA.innerText = 'v';
		}

		this.IsClosedCaptioningVisible = !this.IsClosedCaptioningVisible;
	}

	this.Debug = function(msg)
	{
		SfDebug.DPF(this.m_debugLevel, "SamiDropDownArea: " + msg);
	}
}

// ENDFILE SamiDropDownArea.js ---------------------------------------------------------------------->